//@doc      VIEWX HDR CORE
//@module   VIEWX.H - Auxiliary view declarations |
//  
//  This file contains the declaration of the auxiliary view classes.
//  
//@normal   Copyright <cp> 1996 Meta Four Software. All rights reserved.

#ifndef __K4VIEW_H__
    #error This file is included by "k4view.h", it cannot be used standalone
#endif

/////////////////////////////////////////////////////////////////////////////
// Declarations in this file

	class c4_Strategy;					// system and file interface
    class c4_Sequence;                  // a collection of rows

    class c4_Reference;                 // refers to the actual data values
        class c4_IntRef;
        class c4_FloatRef;
        class c4_DoubleRef;
        class c4_StringRef;
        class c4_ViewRef;

    class c4_Bytes;                     // not defined here
    class c4_Dependencies;              // not defined here
    class c4_Handler;                   // not defined here
    class c4_Notifier;                  // not defined here

/////////////////////////////////////////////////////////////////////////////
//@class A strategy encapsulates code dealing with file I/O and the system.

class c4_Strategy
{
public:
	c4_Strategy ();
	virtual ~c4_Strategy ();

	virtual bool DataOpen(const char*, bool);

	virtual void DataSeek(ulong);
	virtual int  DataRead(void*, int);
	virtual void DataLoad(void*, int);
	virtual void DataWrite(const void*, int);
	virtual void DataCommit(ulong);

	virtual int  StreamRead(void*, void*, int);
	virtual void StreamWrite(void*, const void*, int);

protected:
	c4_File* _file;

	friend class c4_Storage;
};

/////////////////////////////////////////////////////////////////////////////
//@class A sequence is an abstract base class for views on ranges of records.
//
//  Sequences represent arrays of rows (or indexed collections / tables).
//  Insertion and removal of entries is allowed, but can take linear time.
//  A reference count is maintained to decide when the object should go away.

class c4_Sequence
{
public: 
//@group General
    //@cmember Abstract constructor.
    c4_Sequence ();
    
    //@cmember Compares the specified row with another one.
    virtual int Compare(int, c4_Cursor) const;
    //@cmember Replaces the contents of a specified row.
    virtual void SetAt(int, c4_Cursor);
    //@cmember Remaps the index to an underlying view.
    virtual int RemapIndex(int, const c4_Sequence*) const;
    
    //@cmember Returns a descriptions of the current data structure.
    c4_String Describe() const;

//@group Reference counting
    //@cmember Increments the reference count of this sequence.
    void IncRef();
    //@cmember Decrements the reference count, delete objects when last.
    void DecRef();
    //@cmember Returns the current number of references to this sequence.
    int NumRefs() const;

//@group Adding / removing rows
    //@cmember Returns the current number of rows.
    virtual int Size() const = 0;
    //@cmember Changes number of rows, either by inserting or removing them.
    void Resize(int, int =-1);
    
    //@cmember Inserts one or more rows into this sequence.
    virtual void InsertAt(int, c4_Cursor, int =1);
    //@cmember Removes one or more rows from this sequence.
    virtual void RemoveAt(int, int =1);
    //@cmember Move a row to another position.
    virtual void Move(int, int);

//@group Properties
    //@cmember Returns the id of the N-th property.
    int NthProperty(int) const;
    //@cmember Finds the index of a property, or creates a new entry.
    int PropIndex(int, bool =false);
    
    //@cmember Returns the number of data handlers in this sequence.
    virtual int NumHandlers() const = 0;
    //@cmember Returns a reference to the N-th handler in this sequence.
    virtual c4_Handler& NthHandler(int) const = 0;
    //@cmember Returns the context of the N-th handler in this sequence.
    virtual const c4_Sequence* HandlerContext(int) const = 0;
    //@cmember Adds the specified data handler to this sequence.
    virtual int AddHandler(c4_Property&, c4_Handler*) = 0;

//@group Element access
    //@cmember Retrieves one data item from this sequence.
    virtual bool Get(int, int, c4_Bytes&);
    //@cmember Stores a data item into this sequence.
    virtual void Set(int, int, const c4_Bytes&);
    
//@group Dependency notification
    //@cmember Registers a sequence to receive change notifications.
    void Attach(c4_Sequence* child_);
    //@cmember Unregisters a sequence which received change notifications.
    void Detach(c4_Sequence* child_);
    //@cmember Returns a pointer to the dependencies, or null.
    c4_Dependencies* GetDependencies() const;

    //@cmember Called just before a change is made to the sequence.
    virtual c4_Notifier* PreChange(c4_Notifier& nf_);
    //@cmember Called after changes have been made to the sequence.
    virtual void PostChange(c4_Notifier& nf_);
    
protected:
    virtual ~c4_Sequence ();

public: //! for c4_Table::Sequence setup
    virtual void SetSize(int size_) = 0;

private:
    c4_Sequence (const c4_Sequence&);   // not implemented
    void operator= (const c4_Sequence&); // not implemented
    
    int _refCount;
    c4_Dependencies* _dependencies;
};

/////////////////////////////////////////////////////////////////////////////
//@class A reference is used to get or set typed data, using derived classes.
//@xref <c c4_IntRef>, <c c4_FloatRef>, <c c4_DoubleRef>,
//		<c c4_StringRef>, <c c4_ViewRef>

class c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_Reference (c4_RowRef, int);

    //@cmember Assignment of one data item.
    c4_Reference& operator= (const c4_Reference&);

    //@cmember Retrieves the value of the referenced data item.
    bool GetData(c4_Bytes&) const;
    //@cmember Stores a value into the referenced data item.
    void SetData(const c4_Bytes&) const;
    
private:
    void operator& () const;            // not implemented
    
    c4_Cursor _cursor;
    int _propId;
};

/////////////////////////////////////////////////////////////////////////////

    //@class Used to get or set integer values.
    //@base public | <c c4_Reference>
class c4_IntRef : public c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_IntRef (const c4_Reference&);
    operator long () const;
    c4_IntRef& operator= (long);
};

    //@class Used to get or set floating point values.
    //@base public | <c c4_Reference>
class c4_FloatRef : public c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_FloatRef (const c4_Reference&);
    operator double () const;
    c4_FloatRef& operator= (double);
};

    //@class Used to get or set double precision values.
    //@base public | <c c4_Reference>
class c4_DoubleRef : public c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_DoubleRef (const c4_Reference&);
    operator double () const;
    c4_DoubleRef& operator= (double);
};

    //@class Used to get or set string values.
    //@base public | <c c4_Reference>
class c4_StringRef : public c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_StringRef (const c4_Reference&);
    operator c4_String () const;
    c4_StringRef& operator= (const char*);
};

    //@class Used to get or set view values.
    //@base public | <c c4_Reference>
class c4_ViewRef : public c4_Reference
{
public: //@access Public members
    //@cmember Constructor.
    c4_ViewRef (const c4_Reference&);
    operator c4_View () const;
    c4_ViewRef& operator= (const c4_View&);
};

/////////////////////////////////////////////////////////////////////////////
